<?php

namespace App\Http\Controllers;

use App\Models\FooterBacklink;
use App\Models\FooterAnchor;
use Illuminate\Http\Request;

class FooterBacklinkController extends Controller
{
    public function index()
    {
        $backlinks = FooterBacklink::with('anchors')->get();
        return view('footer_backlinks.index', compact('backlinks'));
    }

    public function create()
    {
        return view('footer_backlinks.create');
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'title' => 'required|string|max:255',
                'website_url' => 'required|url|max:255',
                'anchors.*.anchor_text' => 'required|string|max:255',
                'anchors.*.domain' => 'required|string|max:2550',
            ]);
    
            $backlink = FooterBacklink::create([
                'title' => $request->title,
                'website_url' => $request->website_url,
            ]);
    
            foreach ($request->anchors as $anchor) {
                $backlink->anchors()->create($anchor);
            }
    
            return redirect()->route('footer_backlinks.index')->with('success', 'Footer backlink added successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Error: ' . $e->getMessage());
        }
    }


    public function edit(FooterBacklink $footerBacklink)
    {
        return view('footer_backlinks.edit', compact('footerBacklink'));
    }

    public function update(Request $request, FooterBacklink $footerBacklink)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'anchors.*.anchor_text' => 'required|string|max:255',
            'anchors.*.domain' => 'required|string|max:2550',
        ]);

        $footerBacklink->update(['title' => $request->title]);

        $footerBacklink->anchors()->delete();
        foreach ($request->anchors as $anchor) {
            $footerBacklink->anchors()->create($anchor);
        }

        return redirect()->route('footer_backlinks.index')->with('success', 'Footer backlink updated successfully.');
    }

    public function destroy(FooterBacklink $footerBacklink)
    {
        $footerBacklink->delete();
        return redirect()->route('footer_backlinks.index')->with('success', 'Footer backlink deleted successfully.');
    }
    
    public function getAnchorsByDomain(Request $request)
{
    // Validate the domain input (query parameter)
    $request->validate([
        'domain' => 'required|string|max:2550',
    ]);

    // Get the domain from the request, and remove any protocol (http://, https://)
    $domain = parse_url($request->query('domain'), PHP_URL_HOST) ?: $request->query('domain');

    // Get all backlinks where anchors' domain matches the provided domain
    $backlinks = FooterBacklink::whereHas('anchors', function ($query) use ($domain) {
        $query->where('domain', 'like', '%' . $domain . '%');
    })
    ->with(['anchors' => function($query) use ($domain) {
        // Filter anchors by matching the domain
        $query->where('domain', 'like', '%' . $domain . '%');
    }])
    ->get();

    // Extract website URLs and the first anchor text that matches the domain
    $data = $backlinks->map(function ($backlink) {
        // Get the first anchor text (if any)
        $firstAnchorText = $backlink->anchors->first()->anchor_text ?? null;

        return [
            'website_url' => $backlink->website_url,
            'anchors' => $firstAnchorText, // Just the first anchor text as a string
        ];
    });

    // Return only the anchors_data directly as a JSON array
    return response()->json($data);
}

    

}

